/* --------------------------------------------------------------
  Image.js 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/


'use strict';

import Modal from '../libs/Modal'; 
import Router from '../libs/Router'; 
import ImageApi from '../api/ImageApi'; 
import Environment from '../libs/Environment';

/**
 * Image Widget
 *
 * An instance of this widget serves the "image" settings of a style. The user can select
 * one of the uploaded images, preview them or delete them.
 */
export default class Image {
	/**
	 * Class Constructor 
	 * 
	 * @param {Setting} entity Contains the setting information.
	 * @param {StyleConfiguration} styleConfiguration Reference to the current configuration object.
	 * @param {String[]} templateImages An array with the template uploaded images.
	 */
	constructor(entity, styleConfiguration, templateImages) {
		StyleEdit.Validator.isObject(entity);
		StyleEdit.Validator.isObject(styleConfiguration);
		StyleEdit.Validator.isObject(templateImages);
		
		/**
		 * @type {Setting}
		 */
		this.entity = entity;
		
		/**
		 * @type {StyleConfiguration}
		 */
		this.styleConfiguration = styleConfiguration;
		
		/**
		 * Uploaded Images
		 *
		 * @type {String[]}
		 */
		this.templateImages = templateImages;
		
		/**
		 * Base entity name without the "-url" postfix.
		 *
		 * If the setting follows the "gx-setting-image-url" convention, then the base name will be used to select
		 * widgets that have similar names (e.g. "gx-setting-image-repeat") and show/hide them depending on whether
		 * an image is selected or not.
		 *
		 * @type {String}
		 */
		this.baseEntityName = entity.getName().replace('-url', '');
	}
	
	/**
	 * Initialize the widget.
	 */
	initialize() {
		const $element = $('#' + this.entity.getName());
		const $actions = $element.parents().eq(1).find('.image-actions');
		
		if ($element.find('option[value="' + this.entity.getValue() + '"]').length > 0) {
			$element.val(this.entity.getValue());
		} else {
			$element.val('');
		}
		
		const $target = (Environment.isInternetExplorer()) ? $element.parents().eq(1) : $element.parents().eq(0);
		
		$target.mouseover(() => {
			if ($element.val() === '') {
				$actions.find('.preview, .delete').addClass('disabled');
			}
			$actions.stop(true, true); // stop previous animation (if any)
			$actions.show('slide', {direction: 'up'}, 400);
		});
		
		$target.mouseleave(() => {
			$actions.hide('slide', {direction: 'up'}, 400, () => {
				$actions.find('.preview, .delete').removeClass('disabled');
			});
		});
		
		$element.change(() => {
			if ($element.val() !== '') {
				$actions.find('.preview, .delete').removeClass('disabled');
			} else {
				$actions.find('.preview, .delete').addClass('disabled');
			}
			
			$element.trigger('image:toggle_dependencies');
			
			// Update the style configuration. 
			let found = false;
			
			$.each(this.styleConfiguration.getSettings(), (index, setting) => {
				$.each(setting.getEntries(), (index, entry) => {
					if (entry.getName() === this.entity.getName()) {
						entry.setValue($element.val());
						found = true;
						return false;
					}
				});
				
				if (found) {
					return false;
				}
			});
			
			this.entity.setValue($element.val());
		});
		
		$element.on('image:toggle_dependencies', () => {
			// Show/hide image-dependent settings. 
			const method = $element.val() !== '' ? 'show' : 'hide';
			
			$('[id^=' + this.baseEntityName + ']')
				.not('[id=' + this.entity.getName() + ']')
				.each((index, input) => {
					$(input).parents('.form-group')[method]();
				});
		});
		
		$actions.find('.upload').click(() => {
			Router.load('UploadImageModal', [$element]);
		});
		
		$actions.find('.preview').click(event => {
			if ($(event.currentTarget).hasClass('disabled') || $element.val() === '') {
				return;
			}
			Router.load('PreviewImageModal', [$element]);
		});
		
		$actions.find('.delete').click(event => {
			const $target = $(event.currentTarget);
			
			if ($target.hasClass('disabled')) {
				return;
			}
			
			// Check if other settings are using this image. 
			const occurrences = $('.edit-options .image-list').filter(() => {
				return $target.val() === $element.val() && $target.get(0) !== $element.get(0);
			}).length;
			
			if (occurrences > 0) {
				Modal.message(StyleEdit.Language.translate('title_cannot_delete_image', 'style_edit'),
					StyleEdit.Language.translate('message_cannot_delete_image', 'style_edit'));
				return;
			}
			
			// Prompt for image removal. 
			const title = StyleEdit.Language.translate('title_delete_image', 'style_edit');
			const message = StyleEdit.Language.translate('message_delete_image', 'style_edit');
			
			Modal.prompt(title, message).done(() => {
				const filename = $element.find('option:selected').text();
				ImageApi.delete(filename).done(() => {
					$element
						.val('') // Select the empty value.
						.trigger('change'); // Update the style configuration.
					$('.edit-options .image-list option[value="' + filename + '"]').remove();
				});
			});
		});
		
		// Add setting tags to the parent ".form-group" element.
		if (this.entity.getTags() !== null) {
			const tags = this.entity.getTags().join('|');
			$element.parents('.form-group').attr('data-tags', tags);
		}
		
		$element.trigger('change');
	}
	
	/**
	 * Get Widget HTML
	 *
	 * @return {String}
	 */
	getHtml() {
		const template = $('#widget-image-template').html();
		
		// Prepare the template data.
		const data = {
			name: this.entity.getName(),
			title: StyleEdit.Language.translate(this.entity.getName(), 'template'),
			option_upload: StyleEdit.Language.translate('option_upload', 'style_edit'),
			option_preview: StyleEdit.Language.translate('option_preview', 'style_edit'),
			option_delete: StyleEdit.Language.translate('option_delete', 'style_edit'),
			options: [
				{
					value: ''
				}
			]
		};
		
		$.each(this.templateImages, (index, image) => {
			const entry = {
				value: image
			};
			
			data.options.push(entry);
		});
		
		return Mustache.render(template, data);
	}
}
